/*
 * x86work library for Win32
 * 05.07.24 : Makoto NARA (Mc.N)
 */

/*
 * Copyright (c) 2005 Makoto NARA (Mc.N) All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY Makoto NARA (Mc.N) ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


#include "x86work.h"

#ifndef _NTDEF_
typedef LONG NTSTATUS, *PNTSTATUS;
#define STATUS_SUCCESS	((NTSTATUS)0x00000000L)
#endif

// MSR for Win32 struct
typedef struct _MSR_WIN32{
	DWORD num;
	DWORD reserved;
	DWORD lo;
	DWORD hi;
} MSR_WIN32, *PMSR_WIN32;

typedef enum _DEBUG_CONTROL_CODE {
	DebugSysReadIoSpace = 14,
	DebugSysWriteIoSpace = 15,
	DebugSysReadMsr = 16,
	DebugSysWriteMsr = 17,
	DebugSysReadBusData = 18,
	DebugSysWriteBusData = 19,
} DEBUG_CONTROL_CODE;

// ZwSystemDebugControl
typedef NTSTATUS (NTAPI *PZWSYSTEMDEBUGCONTROL)(
	DEBUG_CONTROL_CODE ControlCode,
    PVOID InputBuffer,
	ULONG InputBufferLength,
	PVOID OutputBuffer,
    ULONG OutputBufferLength,
	PULONG ReturnLength
	);

BOOL
SetPrivilege(
    HANDLE hToken,
    LPCTSTR szPrivilege,
    BOOL bEnablePrivilege
    );




BOOL
_cpuid_w32(
	DWORD idx,
	PCPUID_DATA pcpu
	)
{
	return _cpuids_w32(0,0,idx,pcpu);
}





BOOL
_cpuids_w32(
	DWORD numcpu,
	DWORD cntcpu,
	DWORD idx,
	PCPUID_DATA pcpu
	)
{
	DWORD a, b, c, d;
	BOOL bRet = TRUE;

	if(pcpu==NULL){return FALSE;}
	// init
	ZeroMemory(pcpu, sizeof(CPUID_DATA));

	__try{
	_asm{
		pushad
		mov eax, idx
		cpuid
		mov a, eax
		mov b, ebx
		mov c, ecx
		mov d, edx
		popad
	}
	}__except(EXCEPTION_EXECUTE_HANDLER){
		bRet = FALSE;
	}
	if( bRet ){
		pcpu->numcpu = numcpu;
		pcpu->cntcpu = cntcpu;
		pcpu->idx = idx;
		pcpu->eax = a;
		pcpu->ebx = b;
		pcpu->ecx = c;
		pcpu->edx = d;
	}

	return bRet;
}




BOOL
_rdmsr_w32(
	DWORD idx,
	PMSR_DATA pmsr
	)
{
	BOOL bRet = FALSE;
	HMODULE hDLL = NULL;
	PZWSYSTEMDEBUGCONTROL pfnZwSystemDebugControl = NULL;
	HANDLE hToken = NULL;
	MSR_WIN32 msrw32;
	NTSTATUS ntstat;

	// init
	if( pmsr == NULL ){ return FALSE; }
	ZeroMemory(pmsr, sizeof(MSR_DATA));
	ZeroMemory(&msrw32, sizeof(MSR_WIN32));
	bRet = FALSE;

	hDLL = LoadLibrary(TEXT("ntdll.dll"));
	if( hDLL == NULL ){ goto cleanup; }

	pfnZwSystemDebugControl = (PZWSYSTEMDEBUGCONTROL)GetProcAddress(hDLL, TEXT("ZwSystemDebugControl"));
	if( pfnZwSystemDebugControl == NULL ){ goto cleanup; }

	// open Token
	if( !OpenProcessToken(GetCurrentProcess(), TOKEN_ADJUST_PRIVILEGES | TOKEN_QUERY, &hToken) ){ goto cleanup; }

	// enable SeDebugPrivilege
	if( !SetPrivilege(hToken, SE_DEBUG_NAME, TRUE) ){ goto cleanup;	}

	// rdmsr
	msrw32.num = idx;
	ntstat = pfnZwSystemDebugControl( DebugSysReadMsr, &msrw32, sizeof(msrw32), NULL, 0, NULL );
	if( STATUS_SUCCESS == ntstat ){
		pmsr->edx = msrw32.hi;
		pmsr->eax = msrw32.lo;
		bRet = TRUE;
	}

	// disable SeDebugPrivilege
	SetPrivilege(hToken, SE_DEBUG_NAME, FALSE);

cleanup:
	if(NULL!=hToken){CloseHandle(hToken);}
	if(NULL!=hDLL){FreeLibrary(hDLL);}

	return bRet;
}


// [HOWTO] SeDebugPrivilege gpĔCӂ̃vZXւ̃nh擾@
// MSKB : 131065
// http://support.microsoft.com/default.aspx?scid=kb;ja;131065
BOOL
SetPrivilege(
    HANDLE hToken,
    LPCTSTR szPrivilege,
    BOOL bEnablePrivilege
    )
{
    TOKEN_PRIVILEGES tp;
    LUID luid;
    TOKEN_PRIVILEGES tpPrevious;
    DWORD cbPrevious=sizeof(TOKEN_PRIVILEGES);
	BOOL bRet = FALSE;

    if( !LookupPrivilegeValue( NULL, szPrivilege, &luid ) ){
		return FALSE;
	}

    // get current privilege setting
    tp.PrivilegeCount           = 1;
    tp.Privileges[0].Luid       = luid;
    tp.Privileges[0].Attributes = 0;

    bRet = AdjustTokenPrivileges(
            hToken,
            FALSE,
            &tp,
            sizeof(TOKEN_PRIVILEGES),
            &tpPrevious,
            &cbPrevious
            );

	if( !bRet ){ return FALSE; }
    
	// set privilege based on previous setting
    tpPrevious.PrivilegeCount       = 1;
    tpPrevious.Privileges[0].Luid   = luid;

    if(bEnablePrivilege) {
        tpPrevious.Privileges[0].Attributes |= (SE_PRIVILEGE_ENABLED);
    }else{
        tpPrevious.Privileges[0].Attributes ^= (SE_PRIVILEGE_ENABLED & tpPrevious.Privileges[0].Attributes);
    }

    bRet = AdjustTokenPrivileges(
            hToken,
            FALSE,
            &tpPrevious,
            cbPrevious,
            NULL,
            NULL
            );

    return bRet;
}




BOOL
IsSupportedNXOS(
	void
	)
{
	OSVERSIONINFOEX osvie;
	ZeroMemory(&osvie, sizeof(OSVERSIONINFOEX));

	osvie.dwOSVersionInfoSize = sizeof(OSVERSIONINFOEX);
	if( !GetVersionEx((POSVERSIONINFO)&osvie) ){
		return FALSE;
	}

	// NT base?
	if( osvie.dwPlatformId != VER_PLATFORM_WIN32_NT ){ return FALSE; }
	// above xp sp2? ---> SUPPORT!
	if( (osvie.dwMajorVersion==5) && (osvie.dwMinorVersion==1) && (osvie.wServicePackMajor>=2) ){
		return TRUE;
	}
	// above w2k3 sp1? ---> SUPPORT!
	if( (osvie.dwMajorVersion==5) && (osvie.dwMinorVersion==2) && (osvie.wServicePackMajor>=1) ){
		return TRUE;
	}
	//  Major Version is above 6? ---> SUPPORT, maybe...
	if( osvie.dwMajorVersion>5 ){
		return TRUE;
	}
	// Major=5, Minor>2  ---> SUPPORT, maybe...
	if( (osvie.dwMajorVersion==5) && (osvie.dwMinorVersion>2) ){
		return TRUE;
	}

	// w2k or xp sp1
	return FALSE;
}




typedef int (*PFUNC)(void);
BOOL
IsFindOV(
	void
	)
{
	BOOL bRet = FALSE;
	PFUNC pfunc = NULL;
	PBYTE pf = NULL;
	DWORD dwProt = 0;

	// Read/Write/Execute
	pf = (PBYTE)VirtualAlloc( NULL, 4, MEM_COMMIT, PAGE_EXECUTE_READWRITE );
	if( pf == NULL ){ return FALSE; }
	*pf     = 0xC3;	// ret*4
	*(pf+1) = 0xC3;
	*(pf+2) = 0xC3;
	*(pf+3) = 0xC3;

	pfunc = (PFUNC)pf;

	// SEH
	__try{
		(*pfunc)();
	}__except(EXCEPTION_EXECUTE_HANDLER){
		// unknown...
		goto cleanup;
	}

	// Read/Write, NO Execute
	if( VirtualProtect(pf, 4, PAGE_READWRITE, &dwProt) ){
		__try{
			(*pfunc)();
		}__except(EXCEPTION_EXECUTE_HANDLER){
			// work NX-bit!
			bRet = TRUE;
		}
	}

cleanup:
	if( pf != NULL ){ VirtualFree( pf, 0, MEM_DECOMMIT | MEM_RELEASE ); }
	return bRet;
}








