/*
 * x86work library
 * 05.07.24 : Makoto NARA (Mc.N)
 */

/*
 * Copyright (c) 2005 Makoto NARA (Mc.N) All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY Makoto NARA (Mc.N) ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "x86work.h"

BOOL
CloseCPUIDdata(
	PCPUIDS_DATA pcpus
	);

BOOL
initBrandStrings(
	PCPUID_INFOR pci
	);

BOOL
initVendorID(
	PCPUID_INFOR pci
	);

PCPUIDS_DATA
searchCPUIDdata(
	PCPUIDS_DATA pcpus,
	DWORD cntcpu,
	DWORD idx
	);


#ifdef WIN32

PCPUID_INFOR
GetCPUIDdata(
	void
	);

PCPUIDS_DATA
EnumCPUIDs(
	DWORD numcpu,
	DWORD cntcpu
	);

PCPUIDS_DATA
EnumCPUID(
	DWORD numcpu,
	DWORD cntcpu,
	DWORD idx
	);

#endif // WIN32

void
DumpCPUIDdata(
	PCPUID_INFOR pci
	);


BOOL
_cpuid(
	DWORD idx,
	PCPUID_DATA pcpu
	)
{
#ifdef WIN32
	return _cpuid_w32(idx, pcpu);
#endif // WIN32
}


BOOL
_rdmsr(
	DWORD idx,
	PMSR_DATA pmsr
	)
{
#ifdef WIN32
	return _rdmsr_w32(idx, pmsr);
#endif // WIN32
}




BOOL
CloseCPUIDinfo(
	PCPUID_INFOR pci
	)
{
	if( NULL == pci ){ return TRUE; }
	if( NULL != pci->pcpus ){
		CloseCPUIDdata(pci->pcpus);
		pci->pcpus = NULL;
	}
	free(pci);
	
	return TRUE;
}


BOOL
CloseCPUIDdata(
	PCPUIDS_DATA pcpus
	)
{
	PCPUIDS_DATA pnew, p = NULL;

	if( NULL == pcpus ){ return TRUE; }
	p = pcpus;
	for(;;){
		if( NULL == p->pnext ){ free(p); break; }
		pnew = p->pnext;
		free(p);
		p = pnew;
	}

	return TRUE;
}


PCPUID_INFOR
OpenCPUIDinfo(
	void
	)
{
	PCPUID_INFOR pci = NULL;
	BOOL bRet = FALSE;

	// get cpuid information structure
	if( NULL == (pci = GetCPUIDdata()) ){ return NULL; }
	// get vendor id
	initVendorID(pci);
	// get Brand Strings
	initBrandStrings(pci);

	return pci;
}




BOOL
initBrandStrings(
	PCPUID_INFOR pci
	)
{
	PCPUIDS_DATA pcpus = NULL;
	PBYTE pb = NULL;

	if( NULL==pci || NULL==pci->pcpus ){ return FALSE; }

	// init
	pci->szBrand[0] = '\0';
	pci->szBrand[MAX_BRANDSTRING-1] = '\0';
	pb = pci->szBrand;

	// search CPUID(EXTCPUID+4)
	if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, 0, EXTCPUID+4)) ){ return FALSE; }
	*((DWORD*)(pb+32)) = pcpus->cpu.eax;
	*((DWORD*)(pb+36)) = pcpus->cpu.ebx;
	*((DWORD*)(pb+40)) = pcpus->cpu.ecx;
	*((DWORD*)(pb+44)) = pcpus->cpu.edx;
	// search CPUID(EXTCPUID+3)
	if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, 0, EXTCPUID+3)) ){ return FALSE; }
	*((DWORD*)(pb+16)) = pcpus->cpu.eax;
	*((DWORD*)(pb+20)) = pcpus->cpu.ebx;
	*((DWORD*)(pb+24)) = pcpus->cpu.ecx;
	*((DWORD*)(pb+28)) = pcpus->cpu.edx;
	// search CPUID(EXTCPUID+2)
	if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, 0, EXTCPUID+2)) ){ return FALSE; }
	*((DWORD*)(pb+0))  = pcpus->cpu.eax;
	*((DWORD*)(pb+4))  = pcpus->cpu.ebx;
	*((DWORD*)(pb+8))  = pcpus->cpu.ecx;
	*((DWORD*)(pb+12)) = pcpus->cpu.edx;

	return TRUE;
}




BOOL
initVendorID(
	PCPUID_INFOR pci
	)
{
	PCPUIDS_DATA pcpus = NULL;
	PBYTE pb = NULL;

	if( NULL==pci || NULL==pci->pcpus ){ return FALSE; }

	// init
	pci->dwVendor      = CPUVENDOR_UNKNOWN;
	pci->szVendorID[0] = '\0';
	pb = pci->szVendorID;

	// search CPUID(0)
	if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, 0, 0)) ){ return FALSE; }

	*((DWORD*)(pb+0)) = pcpus->cpu.ebx;
	*((DWORD*)(pb+4)) = pcpus->cpu.edx;
	*((DWORD*)(pb+8)) = pcpus->cpu.ecx;
	*(pb+12) = '\0';

	if( 0 == strcmp(pb, "GenuineIntel") ){
		// INTEL
		pci->dwVendor = CPUVENDOR_INTEL;
	}else if( 0 == strcmp(pb, "AuthenticAMD") ){
		// AMD
		pci->dwVendor = CPUVENDOR_AMD;
	}else{
		// unknown...
		pci->dwVendor = CPUVENDOR_UNKNOWN;
	}

	return TRUE;
}




PCPUIDS_DATA
searchCPUIDdata(
	PCPUIDS_DATA pcpus,
	DWORD cntcpu,			// processor #
	DWORD idx				// CPUID(?)
	)
{
	PCPUIDS_DATA p = NULL;

	p = pcpus;

	for(;;){
		if( NULL==p ){ return NULL; }
		if( cntcpu == p->cpu.cntcpu && idx == p->cpu.idx ){
			return p;
		}
		p = p->pnext;
	}
	return NULL;
}


#ifdef WIN32

PCPUID_INFOR
GetCPUIDdata(
	void
	)
{
	HANDLE hCP;
	DWORD pamask, samask, patmp = 0;
	int cntcpu, numcpu = 0;
	BOOL bRet = FALSE;
	PCPUID_INFOR pci = NULL;
	PCPUIDS_DATA p, ptmp;

	if( NULL == (pci = (PCPUID_INFOR)malloc(sizeof(CPUID_INFOR))) ){ goto cleanup; }
	memset(pci, '\0', sizeof(CPUID_INFOR) );

	// How many processors does it have?
	hCP = GetCurrentProcess();
	if( !GetProcessAffinityMask( hCP, &pamask, &samask ) ){ goto cleanup; }
	if( 0==pamask ){ goto cleanup; }

	numcpu = 0; cntcpu = 0;
	bRet = FALSE;
	for(;;){
		patmp = pamask>>numcpu;
		// no more cpu?
		if( 0 == patmp ){
			bRet = TRUE;
			break;
		}
		// 
		if( !(patmp&0x1) ){
			numcpu++;
			continue;
		}

		// set process#
		if( !SetProcessAffinityMask(hCP, 1<<numcpu) ){
			break;
		}
		Sleep(1);
		
		// get CPUIDs
		if( NULL== (p = EnumCPUIDs(numcpu, cntcpu)) ){ break; }
		if( NULL == pci->pcpus ){
			pci->pcpus = p;
		}else{
			ptmp = pci->pcpus;
			for(;;){
				if( ptmp->pnext == NULL ){
					ptmp->pnext = p;
					break;
				}
				ptmp = ptmp->pnext;
			}
		}
		// count cpu!
		cntcpu++;
		numcpu++;
	}
	if( bRet ){
		pci->dwProcessor = cntcpu;
	}

cleanup:
	if(!bRet){
		if( NULL != pci ){
			CloseCPUIDinfo(pci);
		}
		pci = NULL;
	}

	return pci;
}




PCPUIDS_DATA
EnumCPUIDs(
	DWORD numcpu,
	DWORD cntcpu
	)
{
	PCPUIDS_DATA p, ptmp, porg =  NULL;
	DWORD dw, dwValue = 0;
	BOOL bRet = FALSE;

	// get CPUID(0)
	if( NULL == (p = EnumCPUID(numcpu, cntcpu, 0)) ){ goto cleanup; }
	porg = p;

	dwValue = p->cpu.eax;
	for( dw=1; dw<(dwValue+1); dw++ ){
		if( NULL == (ptmp = EnumCPUID(numcpu, cntcpu, dw)) ){ goto cleanup; }
		p->pnext = ptmp;
		p = ptmp;
	}

	// get extended CPUID
	if( NULL == (ptmp = EnumCPUID(numcpu, cntcpu, EXTCPUID+0)) ){ goto cleanup; }
	dwValue = ptmp->cpu.eax;
	if( !(dwValue>=0x80000000) ){
		// not supported extended CPUID
		CloseCPUIDdata(ptmp);
		bRet = TRUE;
		goto cleanup;
	}
	// connect!
	p->pnext = ptmp;
	p = ptmp;
	for( dw=1; dw<(dwValue-EXTCPUID+1); dw++ ){
		if( NULL == (ptmp = EnumCPUID(numcpu, cntcpu, EXTCPUID+dw)) ){ goto cleanup; }
		p->pnext = ptmp;
		p = ptmp;
	}
	bRet = TRUE;

cleanup:
	if(!bRet){
		if(NULL!=porg){ CloseCPUIDdata(porg); }
		porg = NULL;
	}

	return porg;
}


PCPUIDS_DATA
EnumCPUID(
	DWORD numcpu,
	DWORD cntcpu,
	DWORD idx
	)
{
	PCPUIDS_DATA p =  NULL;
	if( NULL == (p = (PCPUIDS_DATA)malloc(sizeof(CPUIDS_DATA))) ){ return NULL; }
	memset(p,0,sizeof(CPUIDS_DATA));
	
	if( !_cpuids_w32(numcpu,cntcpu,idx,&(p->cpu)) ){
		free(p);
		p = NULL;
	}
	return p;
}

#endif // WIN32


BOOL
IsFeature(
	PCPUID_INFOR pci,
	DWORD cntcpu,
	DWORD feature
	)
{
	PCPUIDS_DATA pcpus = NULL;
	BOOL bRet = FALSE;

	if( NULL==pci ){return FALSE;}

	switch(feature){
	case FLAG_CPUID_FPU:	// I`bv̕_Zjbg
	case FLAG_CPUID_VME:	// z[hg
	case FLAG_CPUID_DE:		// fobO@\g
	case FLAG_CPUID_PSE:	// y[WETCYg
	case FLAG_CPUID_TSC:	// ^CEX^vEJE^ (RDTSC)
	case FLAG_CPUID_MSR:	// fŗLWX^ (RDMSR/WRMSR)
	case FLAG_CPUID_PAE:	// AhXg (Ɍ PAE g)
	case FLAG_CPUID_MCE:	// }V`FbNO
	case FLAG_CPUID_CX8:	// CMPXCHG8
	case FLAG_CPUID_APIC:	// on-chip APIC
	case FLAG_CPUID_RSV1:	// reserved(intel)
	case FLAG_CPUID_SEP:	// VXeER[ (SYSENTER/SYSEXIT)
	case FLAG_CPUID_MTRR:	// E^CvEWEWX^
	case FLAG_CPUID_PGE:	// y[WEO[oECl[u
	case FLAG_CPUID_MCA:	// }VE`FbNEA[LeN`
	case FLAG_CPUID_CMOV:	// tړ߂̃T|[g (CMOV)
	case FLAG_CPUID_PAT:	// y[We[u
	case FLAG_CPUID_PSE36:	// 36-bit page size g
	case FLAG_CPUID_PSN:	// Processor Serial Number
	case FLAG_CPUID_CLFSH:	// CLFLUSH
	case FLAG_CPUID_RSV2:	// reserved(intel)
	case FLAG_CPUID_DS:		// Debug Store
	case FLAG_CPUID_ACPI:	// xj^у\tgEFA䂳NbN@\̃T|[g
	case FLAG_CPUID_MMX:	// MMX
	case FLAG_CPUID_FXSR:	// _ReLXg̍Z[u^XgA
	case FLAG_CPUID_SSE:	// SSE
	case FLAG_CPUID_SSE2:	// SSE2
	case FLAG_CPUID_SS:		// ZtEXk[v
	case FLAG_CPUID_HTT:	// Hyper-Threading
	case FLAG_CPUID_TM:		// xj^̃T|[g
	case FLAG_CPUID_RSV3:	// reserved(intel)
	case FLAG_CPUID_PBE:	// Pending Break
		// CPUID(1) and EDX (for Intel Standard)
		if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, cntcpu, 1)) ){ return FALSE; }
		bRet = pcpus->cpu.edx&(1<<(feature&FLAG_CPUID_BITSHIFT)) ? TRUE : FALSE;
		break;
	case FLAG_CPUID_SSE3:		// SSE3
	case FLAG_CPUID_RSV4:		// reserved(intel)
	case FLAG_CPUID_RSV5:		// reserved(intel)
	case FLAG_CPUID_MONITOR:	// MONITOR/MWAIT
	case FLAG_CPUID_DS_CPL:		// CPL Qualified Debug Store
	case FLAG_CPUID_RSV6:	
	case FLAG_CPUID_RSV7:	
	case FLAG_CPUID_EST:		// Intel SpeedStep Technology
	case FLAG_CPUID_TM2:		// xj^Q
	case FLAG_CPUID_RSV8:	
	case FLAG_CPUID_CID:		// Context ID
	case FLAG_CPUID_RSV9:	
	case FLAG_CPUID_RSV10:	
	case FLAG_CPUID_CX16:		// CMPXCHG16B
	case FLAG_CPUID_xTPR:		// Send Task Priority Messages
		// CPUID(1) ECX (for Intel Standard)
		if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, cntcpu, 1)) ){ return FALSE; }
		bRet = pcpus->cpu.ecx&(1<<(feature&FLAG_CPUID_BITSHIFT)) ? TRUE : FALSE;
		break;
	case FLAG_CPUID_SYSCALL:	// SYSCALL/SYSRET (Intel)
	case FLAG_CPUID_XDbit:		// NX-bit/XD-bit (AMD/Intel)
	case FLAG_CPUID_AMDMMX:		// AMD extensions MMX (AMD)
	case FLAG_CPUID_FFXSR:		// Fast FXSAVE and FXRSTOR (AMD)
	case FLAG_CPUID_EM64T:		// AMD64/EM64T (AMD/Intel)
	case FLAG_CPUID_3DNOWEXT:	// Extensions to 3DNow! (AMD)
	case FLAG_CPUID_3DNOW:		// 3DNow! (AMD)
		// extended CPUID(80000001) EDX (for Intel Standard)
		if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, cntcpu, EXTCPUID+1)) ){ return FALSE; }
		bRet = pcpus->cpu.edx&(1<<(feature&FLAG_CPUID_BITSHIFT)) ? TRUE : FALSE;
		break;
	case FLAG_CPUID_LAHF4L:		// LAHF/SAHF for Long mode
	case FLAG_CPUID_CMPLegacy:  // CMPLegacy
	case FLAG_CPUID_LAHF:		// LAHF/SAHF

		// extended CPUID(80000001) ECX (for Intel Standard)
		if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, cntcpu, EXTCPUID+1)) ){ return FALSE; }
		bRet = pcpus->cpu.ecx&(1<<(feature&FLAG_CPUID_BITSHIFT)) ? TRUE : FALSE;
		break;
	default:
		bRet = FALSE;
		break;
	}

	return bRet;
}



DWORD
GetProcessorNum(
	PCPUID_INFOR pci
	)
{
	if( NULL == pci ){ return 0; }
	return pci->dwProcessor;
}


// CPUVENDOR_INTEL, CPUVENDOR_AMD, and CPUVENDOR_UNKNOWN
DWORD
GetCPUVendor(
	PCPUID_INFOR pci
	)
{
	if( NULL == pci ){ return CPUVENDOR_UNKNOWN; }
	return pci->dwVendor;
}

PBYTE
GetVendorID(
	PCPUID_INFOR pci
	)
{
	if( NULL == pci ){ return NULL; }
	return pci->szVendorID;
}

PBYTE
GetVendorString(
	PCPUID_INFOR pci
	)
{
	if( NULL == pci ){ return NULL; }
	return pci->szBrand ;
}



BOOL
GetProcessorData(
	IN  PCPUID_INFOR pci,
	IN  DWORD cntcpu,
	IN  DWORD ind,
	OUT PDWORD pdw
	)
{
	PCPUIDS_DATA pcpus = NULL;

	if( NULL == pci || NULL == pdw ){ return FALSE; }

	switch(ind){
/* in:EAX=1, out=EBX */
	case CPUID_LOCAL_APIC:
		// in  : EAX = 1
		// out : EBX bit 31-24 : Processor local APIC physical ID (Valid for Pentium 4 and subsequent processors)
		if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, cntcpu, 1)) ){ return FALSE; }
		*pdw = (pcpus->cpu.ebx&0xFF000000)>>24;
		break;
	case CPUID_LOGICAL_PROCESSOR_NUM:
		// in  : EAX = 1
		// out : EBX bit 23-16 : Count of logical processors (Valid only Hyper-Threading Technology flag is set)
		if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, cntcpu, 1)) ){ return FALSE; }
		*pdw = (pcpus->cpu.ebx&0x00FF0000)>>16;
		break;
	case CPUID_BRAND_INDEX:
		// in  : EAX = 1
		// out : EBX bit 7-0 : Brand Index - Not supported if the value = 00h
		if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, cntcpu, 1)) ){ return FALSE; }
		*pdw = (pcpus->cpu.ebx&0xFF);
		break;
/* in:EAX=1, out=EAX */
	case CPUID_FAMILY_CODE:
		// in  : EAX = 1
		// out : EAX bit 11-8 : Family Code
		if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, cntcpu, 1)) ){ return FALSE; }
		*pdw = (pcpus->cpu.eax&0x0F00)>>8;
		break;
	case CPUID_MODEL_NUMBER:
		// in  : EAX = 1
		// out : EAX bit 7-4 : Model Number
		if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, cntcpu, 1)) ){ return FALSE; }
		*pdw = (pcpus->cpu.eax&0xF0)>>4;
		break;
	case CPUID_STEPPING_ID:
		// in  : EAX = 1
		// out : EAX bit 3-0 : Stepping ID
		if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, cntcpu, 1)) ){ return FALSE; }
		*pdw = (pcpus->cpu.eax&0xF);
		break;
	case CPUID_PROCESSOR_TYPE:
		// in  : EAX = 1
		// out : EAX bit 13-12 : Processor Type (00:Original OEM Processor, 01:OverDrive Processor, 10:Dual Processor, 11:Reserved.)
		if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, cntcpu, 1)) ){ return FALSE; }
		*pdw = (pcpus->cpu.eax&0x3000)>>12;
		break;
	case CPUID_EXTENDED_MODEL:
		// in  : EAX = 1
		// out : EAX bit 19-16 : Extended Model
		if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, cntcpu, 1)) ){ return FALSE; }
		*pdw = (pcpus->cpu.eax&0xF0000)>>16;
		break;
	case CPUID_EXTENDED_FAMILY:
		// in  : EAX = 1
		// out : EAX bit 27-20 : Extended Family
		if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, cntcpu, 1)) ){ return FALSE; }
		*pdw = (pcpus->cpu.eax&0x0FF00000)>>20;
		break;
	default:
		return FALSE;
	}

	return TRUE;
}

DWORD
GetMultiCoreNum(
	IN  PCPUID_INFOR pci,
	IN  DWORD cntcpu
	)
{
	DWORD num = 0;
	PCPUIDS_DATA pcpus = NULL;

	if( NULL == pci ){ return 0; }

	switch(GetCPUVendor(pci)){
	case CPUVENDOR_INTEL:
		// in  : EAX = 4
		// out : EAX bit 31-26 : Number of processor cores on this die (Multicore)
		if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, cntcpu, 4)) ){
			num = 1;
		}else{
			num = (pcpus->cpu.eax&0xFC000000)>>26;
			num++;
		}
		break;
	case CPUVENDOR_AMD:
		// in  : EAX = EXTCPUID+8)
		// out : ECX bit 7-0 : NC: number of logical CPU cores -1.
		if( NULL == (pcpus = searchCPUIDdata(pci->pcpus, cntcpu, EXTCPUID+8)) ){
			num = 1;
		}else{
			num = (pcpus->cpu.ecx&0xFF);
			num++;
		}
		break;
	default:
		num = 1;
		break;
	}
	return num;
}

BOOL
IsLogicalProcessor(
	IN  PCPUID_INFOR pci,
	IN  DWORD cntcpu
	)
{
	DWORD lapic, llp = 0;
	BOOL bRet = FALSE;
	int i = 0;
	PCPUIDS_DATA pcpus = NULL;

	if( NULL == pci ){ return FALSE; }

	switch(GetCPUVendor(pci)){
	case CPUVENDOR_INTEL:
		bRet = FALSE;
		// Get LocalAPIC and Logical Processor Number.
		if( !GetProcessorData(pci,cntcpu,CPUID_LOCAL_APIC, &lapic) ||
			!GetProcessorData(pci,cntcpu,CPUID_LOGICAL_PROCESSOR_NUM, &llp) ){
			break;
		}
		// HTT enable?
		if( !IsFeature(pci,cntcpu,FLAG_CPUID_HTT) || llp<2 ){
			break;
		}
		// Enable Hyper-Threading Technology
		for(i=0;((DWORD)1<<i)<llp;i++);
		if( lapic&~(0xFFFFFFFF<<i) ){
			bRet = TRUE;
			break;
		}
		break;
	default:
		bRet = FALSE;
		break;
	}
	return bRet;
}


#ifdef _SAMPLE_BUILD
void
DumpCPUIDdata(
	PCPUID_INFOR pci
	)
{
	PCPUIDS_DATA pcpus;
	DWORD cnt = 0xFFFFFFFF;

	if( NULL==pci ){ printf("ERROR: CPUID infor is NULL!\n"); return;}
	
	printf("CPUID Dump sample program\n\n");
	printf("Processor # [%d]\n", GetProcessorNum(pci) );
	printf("dwVendor [%d]\n",    GetCPUVendor(pci) );
	printf("Vendor ID [%s]\n",   GetVendorID(pci)  );
	printf("Brand String [%s]\n",GetVendorString(pci) );
	
	for(pcpus=pci->pcpus; NULL!=pcpus; pcpus=pcpus->pnext){
		if( cnt != pcpus->cpu.cntcpu ){
			cnt = pcpus->cpu.cntcpu;
			printf("==========\n");
			printf("Count CPU = [%d], Process num = [%d]\n", pcpus->cpu.cntcpu, pcpus->cpu.numcpu );
			printf("----------\n");
		}
		printf("idx=[%08X], eax=[%08X], ebx=[%08X], ecx=[%08X], edx=[%08X]\n", pcpus->cpu.idx, pcpus->cpu.eax, pcpus->cpu.ebx, pcpus->cpu.ecx, pcpus->cpu.edx );
	}

	return;
}

int
main(void)
{
	PCPUID_INFOR pci = NULL;
#ifdef __CHECK_MEMORY_LEAK_
_CrtSetDbgFlag ( _CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF );
#endif 
	pci = OpenCPUIDinfo();
	if( NULL==pci ){ return -1; }

//	if( IsFeature(pci,0,FLAG_CPUID_SSE) ){ printf("sse\n"); }
//	DumpCPUIDdata(pci);
	// close
	CloseCPUIDinfo(pci);

#ifdef __CHECK_MEMORY_LEAK_
	_CrtDumpMemoryLeaks();
#endif

	return 0;
}
#endif _SAMPLE_BUILD




