/*
 * Copyright (C) 2011-2012 Makoto NARA (Mc.N)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.synchack.android.usbhostviewer;

import com.synchack.android.usbhostviewer.util.ELog;

import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.database.sqlite.SQLiteStatement;
import android.hardware.usb.UsbConstants;
import android.hardware.usb.UsbDevice;
import android.hardware.usb.UsbEndpoint;
import android.hardware.usb.UsbInterface;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;

public class UsbidsDB {

    // http://www.usb.org/developers/defined_class
    // http://www.linux-usb.org/usb.ids

    private static final String USBIDS_CUR_TABLE = "usbids_cur";
    private static final String USBIDS_CUR_CREATE_TABLE =
            "CREATE TABLE " + USBIDS_CUR_TABLE + " (" +
                    "usbid TEXT," +
                    "name TEXT" +
                    ");";

    private static final String USBIDS_CUR_INSERT_TABLE =
            "INSERT INTO " + USBIDS_CUR_TABLE + " (" +
                    "usbid,name" +
                    ") " +
                    "VALUES (?,?);";

    private static final String USBIDS_CUR_DROP_TABLE =
            "DROP TABLE IF EXISTS " + USBIDS_CUR_TABLE;

    // private static final String[] USBIDS_CUR_QUERY_COL = {
    // "usbid", "name"
    // };

    private final Context ctx;

    public UsbidsDB(Context context) {
        ctx = context;
    }

    private class USBIDsDBHelper extends SQLiteOpenHelper {
        private static final String DATABASE_NAME = "usbids.db";
        private static final int DATABASE_VERSION = 1;

        public USBIDsDBHelper(Context context) {
            super(context, DATABASE_NAME, null, DATABASE_VERSION);
        }

        @Override
        public void onCreate(SQLiteDatabase db) {
            // ELog.i("create db : " + DATABASE_NAME);
            db.execSQL(USBIDS_CUR_CREATE_TABLE);
        }

        @Override
        public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
            // ELog.i("upgrade db : " + DATABASE_NAME);
            db.execSQL(USBIDS_CUR_DROP_TABLE);
            onCreate(db);
        }
    }

    private void makeUSBIDs(SQLiteDatabase db) {

        // db open
        db.beginTransaction();
        SQLiteStatement stmt = db.compileStatement(USBIDS_CUR_INSERT_TABLE);

        BufferedReader bis = null;
        try {
            String line = null;
            String param1 = null;
            String param2 = null;
            String param3 = null;
            String name = null;
            char[] c = new char[3];
            bis = new BufferedReader(new InputStreamReader(ctx.getAssets().open("usb.ids")));
            while (null != (line = bis.readLine())) {
                if (line == null || line.length() < 6) {
                    continue;
                }
                // char c = line.charAt(0);
                for (int i = 0; i < c.length; i++) {
                    c[i] = line.charAt(i);
                }
                if (c[0] == '#') {
                    continue;
                }
                if (c[0] == 'A') {
                    // "AT" is terminate =)
                    break;
                }

                if (c[0] == '\t') {
                    if (param1 == null) {
                        throw new RuntimeException("Unknown parameter : " + line);
                    }
                    char type = param1.charAt(0);
                    if (c[1] == '\t') {
                        if (param2 == null) {
                            throw new RuntimeException("Unknown parameter : " + line);
                        }
                        if (type == 'C') {
                            param3 = line.substring(2, 4);
                            name = line.substring(6);
                            // ELog.d("vid = " + param1 + ", pid = " +
                            // param2 +
                            // ", protocol = "
                            // + param3
                            // + ", name = " + name);
                        } else {
                            throw new RuntimeException("Unknown parameter : " + line);
                        }

                        // add sql
                        addDatabase(stmt, param1, param2, param3, name);
                        continue;
                    }
                    if (type == 'C') {
                        // type class...
                        param2 = line.substring(1, 3);
                        name = line.substring(5);
                        // ELog.d("vid = " + param1 + ", pid = " + param2 +
                        // ", name = " + name);
                    } else {
                        param2 = line.substring(1, 5);
                        name = line.substring(7);
                        // ELog.d("vid = " + param1 + ", pid = " + param2 +
                        // ", name = " + name);
                    }

                    // add sql
                    addDatabase(stmt, param1, param2, "", name);
                    continue;
                }

                param1 = line.substring(0, 4);
                name = line.substring(6);

                // add sqlite
                addDatabase(stmt, param1, "", "", name);
                // ELog.d("vid = " + param1 + ", name = " + name);
            }

            // db finish!
            db.setTransactionSuccessful();

        } catch (IOException e) {
            ELog.e(e.toString());
        } finally {
            if (bis != null) {
                try {
                    bis.close();
                } catch (IOException e1) {
                }
            }
            // db close
            db.endTransaction();
        }
    }

    private void addDatabase(SQLiteStatement stmt, String p1, String p2, String p3, String name) {
        if (p1 == null || p2 == null || p3 == null || name == null) {
            throw new IllegalArgumentException("parameter is null!");
        }
        String tag = p1 + p2 + p3;

        stmt.bindString(1, tag);
        stmt.bindString(2, name == null ? "" : name);
        stmt.executeInsert();

//        ELog.i("tag = " + tag + ", name = " + name);
    }

    public boolean isRemakeDB() {
        USBIDsDBHelper tdn = new USBIDsDBHelper(ctx);
        SQLiteDatabase db = tdn.getReadableDatabase();

        Cursor c = db.query(USBIDS_CUR_TABLE, new String[] {
                "usbid"
        }, null, null, null, null, null);
        c.moveToFirst();
        int size = c.getCount();

        c.close();
        db.close();
        tdn.close();

        return size < 1;
    }

    public void makeDB() {
        USBIDsDBHelper tdn = new USBIDsDBHelper(ctx);
        SQLiteDatabase db = tdn.getWritableDatabase();

        makeUSBIDs(db);

        db.close();
        tdn.close();
    }

    private String getName(String usbid_) {
        String name = "";

        USBIDsDBHelper tdn = new USBIDsDBHelper(ctx);
        SQLiteDatabase db = tdn.getReadableDatabase();

        Cursor c = db.query(USBIDS_CUR_TABLE, new String[] {
                "name"
        }, "usbid == ?", new String[] {
                usbid_
        }, null, null, null);
        c.moveToFirst();
        int size = c.getCount();
        for (int i = 0; i < size; i++) {
            name = c.getString(i);
            if (i > 1) {
                ELog.w("Unknown status... : " + name);
            }
            c.moveToNext();
        }

        c.close();
        db.close();
        tdn.close();

        return name;
    }

    public String getVendorString(UsbDevice udev) {
        return getVendorString(udev.getVendorId());
    }
    
    public String getVendorString(int vid_) {
        return getName(String.format("%04x", vid_));
    }

    public String getProjectString(UsbDevice udev){
        return getProjectString(udev.getVendorId(),udev.getProductId());
    }
    
    public String getProjectString(int vid_, int pid_) {
        return getName(String.format("%04x%04x", vid_, pid_));
    }

    public String getClassString(int cls_) {
        return getName(String.format("C %02x", cls_));
    }

    public String getSubClassString(int cls_, int sub_) {
        return getName(String.format("C %02x%02x", cls_, sub_));
    }

    public String getProtocolString(int cls_, int sub_, int protocol_) {
        return getName(String.format("C %02x%02x%02x", cls_, sub_, protocol_));
    }
    
    
    public String dumpUsbDevice(UsbDevice udev, boolean verbose) {
        String disp = "";

        // int devid = udev.getDeviceId(); // 2002
        String devname = udev.getDeviceName(); // /dev/bus/usb/002/002
        int vid = udev.getVendorId(); // 6353
        int pid = udev.getProductId(); // 19986
        int devclass = udev.getDeviceClass(); // 0
        int subclass = udev.getDeviceSubclass(); // 0
        int protocol = udev.getDeviceProtocol(); // 0

        // (VID):(PIV) productename
        disp = String.format("%04x:%04x ", vid, pid) + getProjectString(vid, pid) + "\n";
        disp += "Device : " + devname + "\n";
        disp += String.format("  Vendor   : 0x%04x ", vid) + getVendorString(vid) + "\n";
        disp += String.format("  Product  : 0x%04x ", pid) + getProjectString(vid, pid)
                + "\n";
        disp += String.format("  Class    : 0x%02x ", devclass) + getClassString(devclass)
                + "\n";
        disp += String.format("  SubClass : 0x%02x ", subclass)
                + getSubClassString(devclass, subclass) + "\n";
        disp += String.format("  Protocol : 0x%02x ", protocol)
                + getProtocolString(devclass, subclass, protocol) + "\n";

        // interface
        int max_if = udev.getInterfaceCount(); // 2
        for (int i = 0; i < max_if; i++) {
            UsbInterface uif = udev.getInterface(i);
            // int if_id = uif.getId(); // 0
            int if_class = uif.getInterfaceClass(); // 8
            int if_proto = uif.getInterfaceProtocol(); // 80
            int if_subclass = uif.getInterfaceSubclass(); // 6

            disp += "\n";
            disp += String.format("Interface (%d/%d)\n", i + 1, max_if);
            disp += String.format("  Class    : 0x%02x ", if_class)
                    + getClassString(if_class) + "\n";
            disp += String.format("  SubClass : 0x%02x ", if_subclass)
                    + getSubClassString(if_class, if_subclass) + "\n";
            disp += String.format("  Protocol : 0x%02x ", if_proto)
                    + getProtocolString(if_class, if_subclass, if_proto) + "\n";

            // endpoint
            int max_ep = uif.getEndpointCount();
            for (int j = 0; j < max_ep; j++) {
                UsbEndpoint uep = uif.getEndpoint(j);

                int ep_addr = uep.getAddress(); // 132
                int ep_dir = uep.getDirection(); // 128
                int ep_enum = uep.getEndpointNumber(); // 4
                //
                // int ep_attr = uep.getAttributes(); // 2
                int ep_inter = uep.getInterval(); // 0
                int ep_maxpacket = uep.getMaxPacketSize(); // 512
                int ep_type = uep.getType();

                disp += "\n";
                disp += String.format("Endpoint (%d/%d)\n", j + 1, max_ep);
                disp += String.format("  Attr.    : 0x%02x (num=%d,", ep_addr, ep_enum)
                        + getEndpointDirString(ep_dir) + ")\n";
                disp += "  Type     : " + getEndpointTypeString(ep_type) + "\n";
                disp += String.format("  Interval : 0x%02x\n", ep_inter);
                disp += String.format("  Packsize : 0x%04x (%d)\n", ep_maxpacket, ep_maxpacket);
            }
        }

        return disp;
    }

    private String getEndpointTypeString(int type_) {
        String name = "(unknown)";

        switch (type_) {
            case UsbConstants.USB_ENDPOINT_XFER_CONTROL:
                name = "Control";
                break;
            case UsbConstants.USB_ENDPOINT_XFER_ISOC:
                name = "Isochronous";
                break;
            case UsbConstants.USB_ENDPOINT_XFER_BULK:
                name = "Bulk";
                break;
            case UsbConstants.USB_ENDPOINT_XFER_INT:
                name = "Interrupt";
                break;
        }

        return name;
    }

    private String getEndpointDirString(int dir_) {
        String name = "(unknown)";

        switch (dir_) {
            case UsbConstants.USB_DIR_OUT:
                name = "OUT";
                break;
            case UsbConstants.USB_DIR_IN:
                name = "IN";
                break;
        }

        return name;
    }

}
