/*
 * Copyright (C) 2011-2012 Makoto NARA (Mc.N)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.synchack.android.usbhostviewer.fragment;

import com.synchack.android.usbhostviewer.R;
import com.synchack.android.usbhostviewer.UsbidsDB;

import android.app.Fragment;
import android.graphics.Typeface;
import android.hardware.usb.UsbConstants;
import android.hardware.usb.UsbDevice;
import android.hardware.usb.UsbEndpoint;
import android.hardware.usb.UsbInterface;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

public class DetailsFragment extends Fragment {

    public static final String TAG_FRAG_DETAILS = "tag_frag_details";

    private UsbidsDB mUsbDB;
    private String mDump = "";

    public static DetailsFragment newInstance(UsbDevice udev) {
        Bundle args = new Bundle();
        args.putParcelable("udev", udev);
        DetailsFragment f = new DetailsFragment();
        f.setArguments(args);
        return f;
    }

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setHasOptionsMenu(true);

        mUsbDB = new UsbidsDB(getActivity());
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container, Bundle savedInstanceState) {
        if (container == null) {
            return null;
        }

        UsbDevice udev = getArguments().getParcelable("udev");
        mDump = dumpUsbDevice(udev, false);

        // make view
        View view = inflater.inflate(R.layout.information, container, false);
        TextView tv = (TextView) view.findViewById(R.id.text);
        tv.setTypeface(Typeface.MONOSPACE);
        tv.setText(mDump);
        return view;
    }

    public String getUsbDump() {
        return mDump;
    }

    //
    // private
    //

    private String dumpUsbDevice(UsbDevice udev, boolean verbose) {
        String disp = "";

        if (udev == null) {
            return disp;
        }

        // int devid = udev.getDeviceId(); // 2002
        String devname = udev.getDeviceName(); // /dev/bus/usb/002/002
        int vid = udev.getVendorId(); // 6353
        int pid = udev.getProductId(); // 19986
        int devclass = udev.getDeviceClass(); // 0
        int subclass = udev.getDeviceSubclass(); // 0
        int protocol = udev.getDeviceProtocol(); // 0

        // (VID):(PIV) productename
        disp = String.format("%04x:%04x ", vid, pid) +
                mUsbDB.getProjectString(vid, pid) + "\n";
        disp += "Device : " + devname + "\n";
        disp += String.format("  Vendor   : 0x%04x ", vid) +
                mUsbDB.getVendorString(vid) + "\n";
        disp += String.format("  Product  : 0x%04x ", pid) +
                mUsbDB.getProjectString(vid, pid)
                + "\n";
        disp += String.format("  Class    : 0x%02x ", devclass) +
                mUsbDB.getClassString(devclass)
                + "\n";
        disp += String.format("  SubClass : 0x%02x ", subclass)
                + mUsbDB.getSubClassString(devclass, subclass) + "\n";
        disp += String.format("  Protocol : 0x%02x ", protocol)
                + mUsbDB.getProtocolString(devclass, subclass, protocol) + "\n";

        // interface
        int max_if = udev.getInterfaceCount(); // 2
        for (int i = 0; i < max_if; i++) {
            UsbInterface uif = udev.getInterface(i);
            // int if_id = uif.getId(); // 0
            int if_class = uif.getInterfaceClass(); // 8
            int if_proto = uif.getInterfaceProtocol(); // 80
            int if_subclass = uif.getInterfaceSubclass(); // 6

            disp += "\n";
            disp += String.format("Interface (%d/%d)\n", i + 1, max_if);
            disp += String.format("  Class    : 0x%02x ", if_class)
                    + mUsbDB.getClassString(if_class) + "\n";
            disp += String.format("  SubClass : 0x%02x ", if_subclass)
                    + mUsbDB.getSubClassString(if_class, if_subclass) + "\n";
            disp += String.format("  Protocol : 0x%02x ", if_proto)
                    + mUsbDB.getProtocolString(if_class, if_subclass, if_proto) + "\n";

            // endpoint
            int max_ep = uif.getEndpointCount();
            for (int j = 0; j < max_ep; j++) {
                UsbEndpoint uep = uif.getEndpoint(j);

                int ep_addr = uep.getAddress(); // 132
                int ep_dir = uep.getDirection(); // 128
                int ep_enum = uep.getEndpointNumber(); // 4
                //
                // int ep_attr = uep.getAttributes(); // 2
                int ep_inter = uep.getInterval(); // 0
                int ep_maxpacket = uep.getMaxPacketSize(); // 512
                int ep_type = uep.getType();

                disp += "\n";
                disp += String.format("Endpoint (%d/%d)\n", j + 1, max_ep);
                disp += String.format("  Attr.    : 0x%02x (num=%d,", ep_addr, ep_enum)
                        + getEndpointDirString(ep_dir) + ")\n";
                disp += "  Type     : " + getEndpointTypeString(ep_type) + "\n";
                disp += String.format("  Interval : 0x%02x\n", ep_inter);
                disp += String.format("  Packsize : 0x%04x (%d)\n", ep_maxpacket,
                        ep_maxpacket);
            }
        }

        return disp;
    }

    private String getEndpointTypeString(int type_) {
        String name = "(unknown)";

        switch (type_) {
            case UsbConstants.USB_ENDPOINT_XFER_CONTROL:
                name = "Control";
                break;
            case UsbConstants.USB_ENDPOINT_XFER_ISOC:
                name = "Isochronous";
                break;
            case UsbConstants.USB_ENDPOINT_XFER_BULK:
                name = "Bulk";
                break;
            case UsbConstants.USB_ENDPOINT_XFER_INT:
                name = "Interrupt";
                break;
        }

        return name;
    }

    private String getEndpointDirString(int dir_) {
        String name = "(unknown)";

        switch (dir_) {
            case UsbConstants.USB_DIR_OUT:
                name = "OUT";
                break;
            case UsbConstants.USB_DIR_IN:
                name = "IN";
                break;
        }

        return name;
    }

}
